<?php
namespace App\Http\Controllers;

use App\Aggregator;
use App\Feeding;
use App\Notifications\NewAggregatorsNotification;
use App\ProgramManager;
use App\State;
use App\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\AggregatorsImport;

class AggregatorController extends Controller
{

    function __construct()
    {
        $this->middleware('permission:aggregator-list|aggregator-create|aggregator-edit|aggregator-delete', [
            'only' => [
                'index',
                'store'
            ]
        ]);
        $this->middleware('permission:aggregator-create', [
            'only' => [
                'create',
                'store'
            ]
        ]);
        $this->middleware('permission:aggregator-edit', [
            'only' => [
                'edit',
                'update'
            ]
        ]);
        $this->middleware('permission:aggregator-delete', [
            'only' => [
                'destroy'
            ]
        ]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if (auth()->user()->hasRole('Admin')) {
            $aggregators = Aggregator::where('is_active', true)->orderBy('id', 'desc')->paginate(50);
        } else {
            $state_name = Auth::user()->state;
            $state = State::where('name', $state_name)->first();

            // $aggregators = Aggregator::where('is_active', true)->where('program_manager_id', Auth::user()->id)->orderBy('id', 'desc')->paginate(50);
            $aggregators = Aggregator::where('is_active', true)->where('state', $state->id)
                ->orderBy('id', 'desc')
                ->paginate(50);
        }
        return view('aggregator.index', compact('aggregators'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $states = State::pluck('name', 'id');
        $program_managers = User::where('user_type', '=', 'State-Manager')->pluck('name', 'id');
        return view('aggregator.create', compact('program_managers', 'states'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $input = $request->all();
        if (Auth::user()->user_type == 'Admin') {
            $state = $input['state'];
        } else {
            $state_name = Auth::user()->state;
            $state = State::where('name', $state_name)->first();
            $state = $state->id;
        }
        
        $input['state'] = $state;
        try {
            
            if(is_array($input['cons_day'])){
                $input['cons_day'] = implode(",", $input['cons_day']);
            }
            
            $aggregator = Aggregator::create($input);

            // get roles wise users
            $users = User::whereHas('roles', function ($q) {
                $q->whereIn('name', [
                    'Admin',
               ]);
            })->get();

            foreach ($users as $user) {
                $user->notify(new NewAggregatorsNotification($aggregator));
            }

            return redirect()->back()->with('success', 'Aggregator successfully created.');
        } catch (Exception $e) {
            return redirect()->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param \App\Aggregator $aggregator
     * @return \Illuminate\Http\Response
     */
    public function show(Aggregator $aggregator)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param \App\Aggregator $aggregator
     * @return \Illuminate\Http\Response
     */
    public function edit(Aggregator $aggregator)
{
    $states = State::pluck('name', 'id');

    // Ensure we convert state name to ID if it's not already numeric
    if (!is_numeric($aggregator->state)) {
        $state = State::where('name', $aggregator->state)->first();
        $aggregator->state = $state?->id; // Use null-safe operator to avoid error
    }

    $aggregator['cons_day'] = explode(",", $aggregator['cons_day']);
    $program_managers = User::where('user_type', '=', 'State-Manager')->pluck('name', 'id');

    return view('aggregator.edit', compact('program_managers', 'aggregator', 'states'));
}


    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param \App\Aggregator $aggregator
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Aggregator $aggregator)
    {
        $input = $request->all();
        try {
             if(is_array($input['cons_day'])){
                $input['cons_day'] = implode(",", $input['cons_day']);
            }
            
            $aggregator->update($input);
            
//             $users = User::whereHas('roles', function ($q) {
//                 $q->whereIn('name', [
//                     'Admin',
//                     'State-Manager'
//                 ]);
//             })->get();
            
            $state = State::where('id', $aggregator->state)->first();
            $users = User::where('user_type', 'State-Manager')->where('state', $state->name)->get();
            
            foreach ($users as $user) {
                $user->notify(new NewAggregatorsNotification($aggregator));
            }
            
            return redirect()->back()->with('success', 'Aggregator successfully updated.');
        } catch (Exception $e) {
            return redirect()->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param \App\Aggregator $aggregator
     * @return \Illuminate\Http\Response
     */
    public function destroy(Aggregator $aggregator)
    {
        try {
            $aggregator->delete();
            return redirect()->back()->with('success', 'Aggregator successfully deleted.');
        } catch (Exception $e) {
            return redirect()->back()
                ->with('error', $e->getMessage())
                ->withInput();
        }
    }

    // aggregators feeding report
    public function feeding_report($id)
    {
        $feedings = Feeding::where('aggregators_id', $id)->where('status', '=', 'Paid')
            ->orderBy('feeding_date', 'asc')
            ->get();
        return view('aggregator.aggregators_feeding_report', compact('feedings'));
    }

    // state wise aggregator list
    public function get_state_wise_aggregator($state)
    {
        $aggregators = Aggregator::where('state', $state)->orderBy('id', 'desc')->paginate(50);
        return view('aggregator.state_wise_aggregator', compact('aggregators'));
    }
    
    public function importAggrs(Request $request){
        $validator = Validator::make($request->all(), [
            'select_file'  => 'required|mimes:xls,xlsx'
        ]);
        
        if($validator->fails()){
            return redirect()->back()->withErrors($validator)->withInput();
        }
        
        Excel::import(new AggregatorsImport, $request->file('select_file'));
        
        return redirect()->back()->with('success', 'Excel Data Imported successfully.');
    }
}
