<?php

namespace App\Http\Controllers;

use App\Cooks;
use App\Feeding;
use App\Helpers\ImageUpload;
use App\Imports\CooksImport;
use App\LocalGovernment;
use App\Notifications\NewCooksNotification;
use App\School;
use App\State;
use App\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
// use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;

class CooksController extends Controller
{

    function __construct()
    {
        $this->middleware('permission:cook-list|cook-create|cook-edit|cook-delete', ['only' => ['index','store']]);
        $this->middleware('permission:cook-create', ['only' => ['create','store']]);
        $this->middleware('permission:cook-edit', ['only' => ['edit','update']]);
        $this->middleware('permission:cook-delete', ['only' => ['destroy']]);
    }

    public function getData(Request $request)
{
    $draw = $request->get('draw');
    $start = $request->get('start');
    $rowPerPage = $request->get('length');

    $orderArray = $request->get('order');
    $columnNameArray = $request->get('columns');
    $searchArray = $request->get('search');

    $columnIndex = $orderArray[0]['column'];
    $columnName = $columnNameArray[$columnIndex]['data'];
    $columnSortOrder = $orderArray[0]['dir'];
    $searchValue = $searchArray['value'];
    $state_id = $request->get('state_id');

    $query = DB::table('cooks')
        ->leftJoin('states', 'states.id', '=', 'cooks.state')
        ->leftJoin('schools', 'schools.id', '=', 'cooks.school_id')
        ->leftJoin('users', 'users.id', '=', 'cooks.user_id')
        ->select(
            'cooks.id',
            'cooks.photo',
            'cooks.first_name',
            'cooks.middle_name',
            'cooks.last_name',
            'cooks.gender',
            'cooks.dob',
            'cooks.phone',
            'cooks.bvn',
            'states.name as state_name',
            'cooks.lga',
            'schools.name as school_name',
            'cooks.students',
            'cooks.status',
            'users.name as added_by'
        );

    if (!auth()->user()->hasRole('Admin')) {
        $query->where('cooks.user_id', Auth::id());
    }

    if (!empty($state_id)) {
        $query->where('cooks.state', $state_id);
    }

    $query->where('cooks.is_active', 1);

    if (!empty($searchValue)) {
        $query->where(function ($q) use ($searchValue) {
            $q->where(DB::raw("CONCAT(cooks.first_name, ' ', cooks.last_name)"), 'like', "%$searchValue%")
              ->orWhere('cooks.bvn', 'like', "%$searchValue%");
        });
    }

    $totalRecords = $query->count();

    $data = $query
        ->orderBy($columnName, $columnSortOrder)
        ->skip($start)
        ->take($rowPerPage)
        ->get();

    return response()->json([
        "draw" => intval($draw),
        "recordsTotal" => $totalRecords,
        "recordsFiltered" => $totalRecords,
        "data" => $data
    ]);
}

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        if(auth()->user()->hasRole('Admin')){
            $cooks  = Cooks::where('is_active', true)->orderBy('id', 'desc')->paginate(500);
            
        }else{
            $state_name = Auth::user()->state;
			$state = State::where('name',$state_name)->first();
		
            $cooks  = Cooks::where('is_active', true)->where('user_id', Auth::user()->id)->orderBy('id', 'desc')->paginate(500);
            // print_r($cooks);
            // exit();
			// $cooks  = Cooks::where('is_active', true)->where('state', $state->id)->orderBy('id', 'desc')->paginate(50);
        }
        return view('cook.index', compact('cooks'));  
    }




    
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $states = State::pluck('name', 'id');
        $state = State::get_state_id(Auth::user()->state);
        if(auth()->user()->hasrole('Admin')){
            $schools = School::where('status', '=', 'Approved')->pluck('name', 'id');
        }else{
            $schools = School::where('state', $state)->where('status', '=', 'Approved')->pluck('name', 'id');
        }
        return view('cook.create', compact('schools', 'states'));   
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $input = $request->all();
        $validator = Validator::make($request->all(),[
            'photo' => 'mimes:jpeg,png,jpg,gif,svg,webp|max:2048',
        ]);
        if($validator->fails()){
            return redirect()->back()->with('error', $validator->errors()->first())->withInput();
        }

        if($request->hasFile('photo')) {
            $file=$request->file('photo');
            $input['photo']=ImageUpload::image_upload($file);
        }

        try{
            $input['user_id'] = Auth::user()->id;
            $cook = Cooks::create($input);
            // get roles wise users
            $users = User::whereHas(
                'roles', function($q){
                    $q->whereIn('name', ['Admin', 'State-Manager']);
                }
            )->get();

            // Notification::send($users, new NewCooksNotification($cook));
            foreach ($users as $user) {
            $user->notify(new NewCooksNotification($cook));
        }

            return redirect()->back()->with('success', 'Cook successfully created.');
        }catch(Exception $e){
            return redirect()->back()->with('error', $e->getMessage())->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Cooks  $cooks
     * @return \Illuminate\Http\Response
     */
    public function show(Cooks $cooks)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Cooks  $cooks
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $cook = Cooks::findOrFail($id);
        $state = State::get_state_id(Auth::user()->state);
        if(auth()->user()->hasrole('Admin')){
            $schools = School::where('status', '=', 'Approved')->pluck('name', 'id');
        }else{
            $schools = School::where('state', $state)->where('status', '=', 'Approved')->pluck('name', 'id');
        }
        $states = State::pluck('name', 'id');
        $lga = LocalGovernment::where('name', '=', $cook->lga)->pluck('name', 'name');
        return view('cook.edit', compact('cook', 'schools','states', 'lga'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Cooks  $cooks
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request,$id)
    {
        $cook = Cooks::findOrFail($id);
        $input = $request->all();
        $validator = Validator::make($request->all(),[
            'photo' => 'mimes:jpeg,png,jpg,gif,svg,webp|max:2048',
        ]);

        if($validator->fails()){
            return redirect()->back()->with('error', $validator->errors()->first())->withInput();
        }

        if($request->hasFile('photo')) {
            $file=$request->file('photo');
            $input['photo']=ImageUpload::image_upload($file);
            ImageUpload::image_delete($cook['photo']);
        }

        try{
            $cook->update($input);
            return redirect()->back()->with('success', 'Cook successfully updated.');
        }catch(Exception $e){
            return redirect()->back()->with('error', $e->getMessage())->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Cooks  $cooks
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
         $cook = Cooks::findOrFail($id);
        try {
            ImageUpload::image_delete($cook['photo']);
            $cook->delete();
            return redirect()->back()->with('success', 'Cook successfully deleted');
        }
        catch(Exception $e){
            return redirect()->back()->with('error', $e->getMessage())->withInput();
        }
    }

    public function importCooks(Request $request){
        $validator = Validator::make($request->all(), [
           'select_file'  => 'required|mimes:xls,xlsx'
          ]);

       if($validator->fails()){
           return redirect()->back()->withErrors($validator)->withInput();
       }
     
         Excel::import(new CooksImport, $request->file('select_file'));

          return redirect()->back()->with('success', 'Excel Data Imported successfully.');
      
   }
       // aggregators feeding report
    public function feeding_report($id){
        $feedings = Feeding::where('cooks_id', $id)->where('status', '=', 'Paid')->orderBy('feeding_date', 'asc')->get();
        return view('cook.cooks_feeding_report', compact('feedings')); 
     }
    // state wise cook list
    public function get_state_wise_cook($state){
        
        
        $cooks = Cooks::where('state', $state)->orderBy('id', 'desc')->paginate(10);
        return view('cook.state_wise_cook', compact('cooks'));
    }
}
