<?php

namespace App\Http\Controllers;

use App\Exports\SchoolExport;
use App\LocalGovernment;
use App\School;
use App\SchoolSponsor;
use App\Imports\SchoolImport;
use App\Notifications\NewSchoolNotification;
use App\State;
use App\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\DB;

class SchoolController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:school-list|school-create|school-edit|school-delete', ['only' => ['index', 'store']]);
        $this->middleware('permission:school-create', ['only' => ['create', 'store']]);
        $this->middleware('permission:school-edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:school-delete', ['only' => ['destroy']]);
    }

    public function suggestions()
    {
        if (auth()->user()->hasrole('Admin')) {
            School::where('seen', '=', 0)->update(['seen' => 1]);
            $schools = School::where('is_sponcered', '=', 'suggested')->orderBy('id', 'DESC')->paginate(10);
        } else {
            $state = State::get_state_id(Auth::user()->state);
            School::where('state', $state)->where('seen', '=', 0)->update(['seen' => 1]);
            $schools = School::where('state', $state)->where('is_sponcered', '=', 'suggested')->orderBy('id', 'DESC')->paginate(50);
        }

        return view('school.suggestions', compact('schools'));
    }

    public function index()
    {
        if (auth()->user()->hasrole('Admin')) {
            $schools = School::orderBy('id', 'DESC')->paginate(10);
        } else {
            $state = State::get_state_id(Auth::user()->state);
            $schools = School::where('state', $state)->orderBy('id', 'DESC')->paginate(10);
        }

        return view('school.index', compact('schools'));
    }

    public function create()
    {
        $states = State::select('id', 'name')->get();
        return view('school.create', compact('states'))->with('school', null);
    }

    public function store(Request $request)
    {
        $input = $request->all();

        try {
            $school = School::create($input);

            $users = User::whereHas('roles', function ($q) {
                $q->whereIn('name', ['Admin', 'State-Manager']);
            })->get();

            foreach ($users as $user) {
                $user->notify(new NewSchoolNotification($school));
            }

            return redirect()->back()->with('success', 'School has been created.');
        } catch (Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function edit(School $school)
    {
        $states = State::select('id', 'name')->get();
        $lga = LocalGovernment::where('name', '=', $school->lga)->pluck('name', 'name');
        return view('school.edit', compact('school', 'states', 'lga'));
    }

    public function update(Request $request, School $school)
    {
        try {
            $school->update($request->all());
            return redirect()->back()->with('success', 'Data has been updated.');
        } catch (Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function destroy(School $school)
    {
        try {
            $school->delete();
            return redirect()->back()->with(['alert-type' => 'success', 'message' => 'Data has been Deleted!']);
        } catch (Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function get_lga(Request $request)
    {
        $search = $request->search;

        $lga = LocalGovernment::orderby('id', 'asc')
            ->select('id', 'name')
            ->when($search, function ($query, $search) {
                return $query->where('name', 'like', "%{$search}%");
            })
            ->where('state_id', $request->state)
            ->get();

        $response = $lga->map(function ($data) {
            return [
                "id" => $data->name,
                "text" => $data->name,
            ];
        });

        return response()->json($response);
    }

    public function exportSchool()
    {
        return Excel::download(new SchoolExport, Carbon::now()->toDateString() . '-school-data.xlsx');
    }

    public function importSchool(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'select_file' => 'required|mimes:xls,xlsx'
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        Excel::import(new SchoolImport, $request->file('select_file'));

        return redirect()->back()->with('success', 'Excel Data Imported successfully.');
    }

    public function get_state_wise_school($state)
    {
        $schools = School::where('state', $state)->orderBy('id', 'desc')->paginate(10);
        return view('school.state_wise_school', compact('schools'));
    }

    public function get_state_school($state)
    {
        $schools = School::where('state', $state)->orderBy('id', 'desc')->paginate(10);
        return view('state_schools', compact('schools'));
    }

    public function get_gov_school(Request $request)
    {
        return view('gov_schools', [
            'gov' => $request->gov,
            'state' => $request->state
        ]);
    }

    public function get_gov_school_ajax(Request $request, $keyword)
    {
        $schools = School::where('lga', $request->gov)
            ->where('name', 'like', "%{$keyword}%")
            ->where('status', '!=', 'Pending')
            ->orderBy('id', 'desc')
            ->get();

        return response()->json($schools);
    }

    public function get_all_school_ajax(Request $request)
    {
        $search = $request->search;

        $schools = School::select('id', 'name')
            ->where('lga', $request->lga)
            ->when($search, function ($query, $search) {
                return $query->where('name', 'like', "%{$search}%");
            })
            ->get();

        $response = $schools->map(function ($data) {
            return [
                "id" => $data->name,
                "text" => $data->name
            ];
        });

        return response()->json($response);
    }

    public function add_school(Request $request)
    {
        $request->validate([
            'editor_name' => 'required',
            'editor_email' => 'required',
            'name' => 'required',
        ]);

        School::create($request->all());

        return back()->with(['type' => 'success', 'message' => 'School suggestion has been added successfully']);
    }

    public function get_state_ajax(Request $request, $keyword)
    {
        $states = State::where('name', 'like', "%{$keyword}%")->orderBy('id', 'desc')->get();
        return response()->json($states);
    }

    public function sponcor()
    {
        $states = State::select('id', 'name')->get();
        $lga = State::pluck('name', 'id');
        $school = School::pluck('name', 'id');
        return view('sponcor_schools', compact('states', 'lga', 'school'));
    }

    public function save_sponcor_school(Request $request)
    {
        $school = School::where('lga', $request->lga)
            ->where('state', $request->state)
            ->where('name', 'like', "%{$request->school_name}%")
            ->first();

        if ($school) {
            $data = [
                'school_id' => $school->id,
                'name' => $request->school_name,
                'contact_person' => $request->contact_person,
                'phone' => $request->phone,
                'email' => $request->email,
                'state' => $request->state,
                'lga' => $request->lga,
                'address' => $request->address,
                'status' => $request->status,
                'seen' => 0,
            ];

            DB::table('schools_sponsors')->insert($data);

            return back()->with(['type' => 'success', 'message' => 'School Sponsor has been added successfully']);
        }

        return back()->with(['type' => 'error', 'message' => 'School not found. Please check the data.']);
    }

    public function sponsored_schools()
    {
        if (auth()->user()->hasrole('Admin')) {
            SchoolSponsor::where('seen', '=', 0)->update(['seen' => 1]);
            $schools = SchoolSponsor::where('status', '=', 'Approved')->orderBy('id', 'DESC')->paginate(10);
        } else {
            $state = State::get_state_id(Auth::user()->state);
            SchoolSponsor::where('state', $state)->where('seen', '=', 0)->update(['seen' => 1]);
            $schools = SchoolSponsor::where('state', $state)->where('status', '=', 'Approved')->orderBy('id', 'DESC')->paginate(50);
        }

        return view('school.sponsored', compact('schools'));
    }

    public function getSchoolsByStateLga(Request $request)
        {
            $schools = \App\School::where('state', $request->state)
                ->where('lga', $request->lga)
                ->select('id', 'name')
                ->get();

            return response()->json($schools);
        }

    
}
