<?php

namespace App\Http\Controllers;

use App\Helpers\ImageUpload;
use App\State;
use App\User;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{

    function __construct()
    {
        $this->middleware('permission:permissions-list|permissions-create|permissions-edit|permissions-delete', ['only' => ['index','store']]);
        $this->middleware('permission:permissions-create', ['only' => ['create','store']]);
        $this->middleware('permission:permissions-edit', ['only' => ['edit','update']]);
        $this->middleware('permission:permissions-delete', ['only' => ['destroy']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
    */

    public function index(Request $request)
    {
        // $data = User::orderBy('id','DESC')->paginate(5);
        $data = User::orderBy('id','DESC')->get();
        return view('users.index',compact('data'));
            // ->with('i', ($request->input('page', 1) - 1) * 5);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = User::find(Auth::id());
        foreach ($user->getRoleNames() as $user_role) {
              $user_role;
        }
        //  $roles = Role::where('name', '!=', $user_role)->pluck('name','name');
         $roles = Role::pluck('name','name');
        $states = State::pluck('name', 'name');
        return view('users.create',compact('roles', 'states'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'name' => 'required',
            'photo' => 'mimes:png,jpg,jpeg,webp|max:2048',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8',
            'roles' => 'required'
        ]);
        
        $input = $request->all();

        if($request->hasFile('photo')){
            $file = $request->file('photo');
            $input['photo'] = ImageUpload::image_upload($file);
       }

        $input['password'] = Hash::make($input['password']);

        $user = User::create($input);
        $user->assignRole($request->input('roles'));

        return redirect()->route('users.index')
            ->with('success','User created successfully');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = User::find($id);
        return view('users.show',compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = User::find($id);
        $authUser = User::find(Auth::id());
        foreach ($authUser->getRoleNames() as $user_role) {
              $user_role;
        }
        //  $roles = Role::where('name', '!=', $user_role)->pluck('name','name');
        $roles = Role::pluck('name','name');
        $states = State::pluck('name', 'name');
        $userRole = $user->roles->pluck('name','name')->all();

        return view('users.edit',compact('user','roles','userRole', 'states'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'name' => 'required',
            'email' => 'required|email|unique:users,email,'.$id,
            // 'password' => 'same:confirm-password',
            // 'password' => 'required|string|min:8',
            'photo' => 'mimes:png,jpg,jpeg,webp|max:2048',
            'roles' => 'required'
        ]);

        $user = User::findOrFail($id);

        $input = $request->all();

        if($request->hasFile('photo')){
            $file = $request->file('photo');
            $input['photo'] = ImageUpload::image_upload($file);
            ImageUpload::image_delete($user['photo']);
       }


        if(!empty($input['password'])){
            $input['password'] = Hash::make($input['password']);
        }else{
            $input = Arr::except($input,array('password'));
        }

        $user = User::find($id);
        $user->update($input);
        DB::table('model_has_roles')->where('model_id',$id)->delete();

        $user->assignRole($request->input('roles'));

        return redirect()->route('users.index')
            ->with('success','User updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        User::find($id)->delete();
        // return redirect()->route('users.index')
        //     ->with('success','User deleted successfully');
        $notification = array(
            'message' => 'Data has been Deleted!',
            'alert_type' => 'success'
        );
        return redirect()->back()->with($notification);
    }

    // get state admin list
    public function state_admin_list(){
           $state_admins = User::where('user_type', '=', 'State-Manager')->get();
           return view('state_admin.state_admin_list', compact('state_admins'));
    }

    // get role wise admin list
    public function get_all_admin($role){
           $state_admins = User::whereHas('roles', function($q) use ($role){
                $q->where('name', '=', $role);
           })->get();
           return view('state_admin.state_admin_list', compact('state_admins'));
    }

     //  password change 
    public function user_profile(){
        $id = Auth::id();
        $user = User::findOrFail($id);
        return view('users.user_profile', compact('user'));
 }

 public function get_pass_change_form(){
    return view('users.change_pass_form');
}
 


   public function save_changed_password(Request $request){
       $this->validate($request, [
           'password' => ['required', 'string', 'min:8', 'confirmed'],
       ]);

       $input = $request->all();
       if(!empty($input['password'])){
           $input['password'] = Hash::make($input['password']);
       }else{
           $input = Arr::except($input,array('password'));
       }

       $id = Auth::user()->id;

       $user = User::find($id);
       try{
           $user->update($input);
           return redirect('/my-profile')->with('success', 'Password Changed Successfully.');
       }catch(Exception $e){
            return redirect()->back()->with('error', $e->getMessage())->withInput();
       }
   }

//    profile picture update
  public function save_profile_picture(Request $request){
           $validator = Validator::make($request->all(),[
               'photo' => 'mimes:png,jpg,jpeg,webp|max:2048'
           ]);
           if($validator->fails()){
               return redirect()->back()->withErrors($validator)->withInput();
           }
           $input = $request->all();
           $userId = Auth::user()->id;
           $user = User::findOrFail($userId);
           
           if($request->hasFile('photo')){
            $file = $request->file('photo');
            $input['photo'] = ImageUpload::image_upload($file);
            ImageUpload::image_delete($user['photo']);
       }
           try{
            $user->update($input);
            return redirect('/my-profile')->with('success', 'Profile Picture Changed Successfully.');
        }catch(Exception $e){
             return redirect()->back()->with('error', $e->getMessage())->withInput();
        }
  }

}
